/*
 * Decompiled with CFR 0.152.
 */
package net.sf.regain.crawler;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import net.sf.regain.RegainException;
import net.sf.regain.RegainToolkit;
import net.sf.regain.crawler.ErrorLogger;
import net.sf.regain.crawler.Profiler;
import net.sf.regain.crawler.UrlChecker;
import net.sf.regain.crawler.config.CrawlerConfig;
import net.sf.regain.crawler.config.UrlMatcher;
import net.sf.regain.crawler.document.DocumentFactory;
import net.sf.regain.crawler.document.RawDocument;
import net.sf.regain.crawler.plugin.CrawlerPluginManager;
import org.apache.log4j.Logger;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.DateTools;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.Term;
import org.apache.lucene.index.TermEnum;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopScoreDocCollector;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;

public class IndexWriterManager {
    private static Logger mLog = Logger.getLogger(IndexWriterManager.class);
    private static final String NEW_INDEX_SUBDIR = "new";
    private static final String QUARANTINE_INDEX_SUBDIR = "quarantine";
    private static final String WORKING_INDEX_SUBDIR = "index";
    private static final String TEMP_INDEX_SUBDIR = "temp";
    private static final String BREAKPOINT_INDEX_SUBDIR = "breakpoint";
    private static final boolean WRITE_TERMS_SORTED = true;
    private static final long RENAME_TIMEOUT = 60000L;
    private static final int WRITING_MODE = 1;
    private static final int READING_MODE = 2;
    private static final int SEARCHING_MODE = 3;
    private static final int ALL_CLOSED_MODE = 4;
    private CrawlerConfig mConfig;
    private Analyzer mAnalyzer;
    private IndexWriter mIndexWriter;
    private IndexReader mIndexReader;
    private IndexSearcher mIndexSearcher;
    private boolean mUpdateIndex;
    private boolean mRetryFailedDocs;
    private DocumentFactory mDocumentFactory;
    private File mNewIndexDir;
    private File mQuarantineIndexDir;
    private File mTempIndexDir;
    private Directory mLuceneTempIndexDir;
    private File mBreakpointIndexDir;
    private File mAnalysisDir;
    private File mErrorLogFile;
    private FileOutputStream mErrorLogStream;
    private PrintWriter mErrorLogWriter;
    private int mInitialDocCount;
    private Profiler mAddToIndexProfiler = new Profiler("Indexed documents", "docs");
    private Profiler mBreakpointProfiler = new Profiler("Created breakpoints", "breakpoints");
    private HashMap<String, String> mUrlsToDeleteHash;
    private CrawlerPluginManager pluginManager = CrawlerPluginManager.getInstance();

    public IndexWriterManager(CrawlerConfig config, boolean updateIndex, boolean retryFailedDocs) throws RegainException {
        boolean createNewIndex;
        this.mConfig = config;
        this.mUpdateIndex = updateIndex;
        this.mRetryFailedDocs = retryFailedDocs;
        this.mInitialDocCount = 0;
        File indexDir = new File(config.getIndexDir());
        if (!indexDir.exists()) {
            mLog.info("Creating index directory " + indexDir.getAbsolutePath());
            if (!indexDir.mkdirs()) {
                throw new RegainException("Could not create index directory!");
            }
        }
        this.mNewIndexDir = new File(indexDir, NEW_INDEX_SUBDIR);
        this.mQuarantineIndexDir = new File(indexDir, QUARANTINE_INDEX_SUBDIR);
        this.mTempIndexDir = new File(indexDir, TEMP_INDEX_SUBDIR);
        try {
            this.mLuceneTempIndexDir = FSDirectory.open(this.mTempIndexDir);
        }
        catch (IOException ioEx) {
            throw new RegainException("Couldn't open tmpIndexDir", ioEx);
        }
        this.mBreakpointIndexDir = new File(indexDir, BREAKPOINT_INDEX_SUBDIR);
        this.mErrorLogFile = new File(this.mTempIndexDir, "log/error.log");
        if (this.mTempIndexDir.exists()) {
            RegainToolkit.deleteDirectory(this.mTempIndexDir);
        }
        if (!this.mTempIndexDir.mkdir()) {
            throw new RegainException("Creating working directory failed: " + this.mTempIndexDir.getAbsolutePath());
        }
        String[] untokenizedFieldNames = config.getUntokenizedFieldNames();
        String analyzerType = config.getAnalyzerType();
        String[] stopWordList = config.getStopWordList();
        String[] exclusionList = config.getExclusionList();
        this.mAnalyzer = RegainToolkit.createAnalyzer(analyzerType, stopWordList, exclusionList, untokenizedFieldNames);
        if (updateIndex && !this.copyExistingIndex(indexDir, analyzerType)) {
            updateIndex = false;
            this.mUpdateIndex = false;
        }
        boolean bl = createNewIndex = !updateIndex;
        if (createNewIndex) {
            try {
                this.mIndexWriter = this.createIndexWriter(true);
            }
            catch (IOException exc) {
                throw new RegainException("Creating new index failed", exc);
            }
        }
        if (updateIndex) {
            this.setIndexMode(2);
            try {
                IndexWriter.unlock(this.mIndexReader.directory());
                this.mInitialDocCount = this.mIndexReader.numDocs();
            }
            catch (IOException exc) {
                throw new RegainException("Forcing unlock failed", exc);
            }
        }
        RegainToolkit.writeToFile(analyzerType, new File(this.mTempIndexDir, "analyzerType.txt"));
        RegainToolkit.writeListToFile(stopWordList, new File(this.mTempIndexDir, "stopWordList.txt"));
        RegainToolkit.writeListToFile(exclusionList, new File(this.mTempIndexDir, "exclusionList.txt"));
        if (untokenizedFieldNames.length != 0) {
            RegainToolkit.writeListToFile(untokenizedFieldNames, new File(this.mTempIndexDir, "untokenizedFieldNames.txt"));
        }
        if (config.getWriteAnalysisFiles()) {
            this.mAnalysisDir = new File(this.mTempIndexDir.getAbsolutePath() + File.separator + "analysis");
            if (!this.mAnalysisDir.mkdir() && !this.mAnalysisDir.exists()) {
                throw new RegainException("Creating analysis directory failed: " + this.mAnalysisDir.getAbsolutePath());
            }
        }
        this.mDocumentFactory = new DocumentFactory(config, this.mAnalysisDir);
    }

    public boolean getUpdateIndex() {
        return this.mUpdateIndex;
    }

    public int getInitialDocCount() {
        return this.mInitialDocCount;
    }

    public int getAddedDocCount() {
        return this.mAddToIndexProfiler.getMeasureCount();
    }

    public int getRemovedDocCount() {
        HashMap<String, String> hash = this.mUrlsToDeleteHash;
        return hash == null ? 0 : hash.size();
    }

    public void logError(String msg, Throwable thr) throws RegainException {
        if (this.mErrorLogStream == null) {
            try {
                new File(this.mTempIndexDir, "log").mkdir();
                this.mErrorLogStream = new FileOutputStream(this.mErrorLogFile, true);
                this.mErrorLogWriter = new PrintWriter(this.mErrorLogStream);
            }
            catch (IOException exc) {
                throw new RegainException("Opening error log file of the index failed", exc);
            }
        }
        if (thr == null) {
            this.mErrorLogWriter.println(msg);
        } else {
            this.mErrorLogWriter.println(msg + ":");
            thr.printStackTrace(this.mErrorLogWriter);
            this.mErrorLogWriter.println();
        }
        this.mErrorLogWriter.flush();
    }

    private void setIndexMode(int mode) throws RegainException {
        if ((mode == 1 || mode == 4) && this.mIndexReader != null) {
            try {
                this.mIndexReader.close();
                this.mIndexReader = null;
            }
            catch (IOException exc) {
                throw new RegainException("Closing IndexReader failed", exc);
            }
        }
        if ((mode == 2 || mode == 4) && this.mIndexWriter != null) {
            try {
                this.mIndexWriter.close();
                this.mIndexWriter = null;
            }
            catch (IOException exc) {
                throw new RegainException("Closing IndexWriter failed", exc);
            }
        }
        if (mode == 4 && this.mIndexSearcher != null) {
            try {
                this.mIndexSearcher.close();
                this.mIndexSearcher = null;
            }
            catch (IOException exc) {
                throw new RegainException("Closing IndexSearcher failed", exc);
            }
        }
        if (mode == 1 && this.mIndexWriter == null) {
            mLog.info("Switching to index mode: adding mode");
            try {
                this.mIndexWriter = this.createIndexWriter(false);
            }
            catch (IOException exc) {
                throw new RegainException("Creating IndexWriter failed", exc);
            }
        }
        if (mode == 2 && this.mIndexReader == null) {
            mLog.info("Switching to index mode: deleting mode");
            try {
                this.mIndexReader = IndexReader.open(this.mLuceneTempIndexDir, false);
            }
            catch (IOException exc) {
                throw new RegainException("Creating IndexReader failed", exc);
            }
        }
        if (mode == 3 && this.mIndexSearcher == null) {
            mLog.info("Switching to index mode: searching mode");
            try {
                this.mIndexSearcher = new IndexSearcher(this.mLuceneTempIndexDir, false);
            }
            catch (IOException exc) {
                throw new RegainException("Creating IndexSearcher failed", exc);
            }
        }
        if (mode == 4) {
            mLog.info("Switching to index mode: all closed mode");
        }
    }

    private IndexWriter createIndexWriter(boolean createNewIndex) throws IOException {
        IndexWriterConfig iConfig = new IndexWriterConfig(RegainToolkit.getLuceneVersion(), this.mAnalyzer);
        if (createNewIndex) {
            iConfig.setOpenMode(IndexWriterConfig.OpenMode.CREATE);
        } else {
            iConfig.setOpenMode(IndexWriterConfig.OpenMode.APPEND);
        }
        IndexWriter indexWriter = new IndexWriter(this.mLuceneTempIndexDir, iConfig);
        int maxFieldLength = this.mConfig.getMaxFieldLength();
        if (maxFieldLength > 0) {
            indexWriter.setMaxFieldLength(maxFieldLength);
        }
        return indexWriter;
    }

    private boolean copyExistingIndex(File indexDir, String analyzerType) throws RegainException {
        File oldIndexDir = this.mBreakpointIndexDir.exists() ? this.mBreakpointIndexDir : (this.mNewIndexDir.exists() ? this.mNewIndexDir : new File(indexDir, WORKING_INDEX_SUBDIR));
        if (!oldIndexDir.exists()) {
            mLog.warn("Can't update index, because there was no old index. A complete new index will be created...");
            return false;
        }
        File analyzerTypeFile = new File(oldIndexDir, "analyzerType.txt");
        String analyzerTypeOfIndex = RegainToolkit.readStringFromFile(analyzerTypeFile);
        if (analyzerTypeOfIndex == null) {
            mLog.warn("Can't update index, because the index was created using an unknown analyzer type, configured type '" + analyzerType + "'). " + "A complete new index will be created...");
            return false;
        }
        if (!analyzerType.equals(analyzerTypeOfIndex.trim())) {
            mLog.warn("Can't update index, because the index was created using another analyzer type (index type: '" + analyzerTypeOfIndex.trim() + "', configured type '" + analyzerType + "'). " + "A complete new index will be created...");
            return false;
        }
        mLog.info("Updating index from " + oldIndexDir.getAbsolutePath());
        RegainToolkit.copyDirectory(oldIndexDir, this.mTempIndexDir, false, ".txt");
        return true;
    }

    public boolean isAlreadyIndexed(String url) throws RegainException {
        boolean result = false;
        if (this.mUpdateIndex) {
            Term urlTerm = new Term("url", url);
            TermQuery query = new TermQuery(urlTerm);
            try {
                this.setIndexMode(3);
                TopScoreDocCollector collector = TopScoreDocCollector.create(2, false);
                this.mIndexSearcher.search((Query)query, collector);
                if (collector.getTotalHits() == 1) {
                    result = true;
                }
            }
            catch (IOException exc) {
                throw new RegainException("Searching old index entry failed for " + url, exc);
            }
        }
        return result;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void addToIndex(RawDocument rawDocument, ErrorLogger errorLogger) throws RegainException {
        if (this.mUpdateIndex) {
            Document doc;
            boolean removeOldEntry = false;
            Term urlTerm = new Term("url", rawDocument.getUrl());
            TermQuery query = new TermQuery(urlTerm);
            try {
                this.setIndexMode(3);
                TopScoreDocCollector collector = TopScoreDocCollector.create(20, false);
                this.mIndexSearcher.search((Query)query, collector);
                ScoreDoc[] hits = collector.topDocs().scoreDocs;
                if (hits.length > 0) {
                    if (hits.length > 1) {
                        for (int i = 1; i < hits.length; ++i) {
                            this.markForDeletion(this.mIndexSearcher.doc(hits[i].doc));
                        }
                        mLog.warn("There are duplicate entries (" + hits.length + " in " + "total) for " + rawDocument.getUrl() + ". They will be removed.");
                    }
                    doc = this.mIndexSearcher.doc(hits[0].doc);
                } else {
                    doc = null;
                }
            }
            catch (IOException exc) {
                throw new RegainException("Searching old index entry failed for " + rawDocument.getUrl(), exc);
            }
            if (doc != null) {
                Date docLastModified = rawDocument.getLastModified();
                if (docLastModified == null) {
                    mLog.info("Don't know when the document was last modified. Creating a new index entry...");
                    removeOldEntry = true;
                } else {
                    String asString = doc.get("last-modified");
                    if (asString != null) {
                        long diff = 86400001L;
                        Date indexLastModified = null;
                        try {
                            indexLastModified = DateTools.stringToDate(asString);
                            diff = docLastModified.getTime() - indexLastModified.getTime();
                        }
                        catch (ParseException parseException) {
                            mLog.warn("Couldn't parse last-modified date from index. Document: " + rawDocument.getUrl(), parseException);
                        }
                        if (diff > 86400000L) {
                            mLog.info("Index entry is outdated. Creating a new one (source=" + docLastModified + "), (index=" + indexLastModified + "): " + rawDocument.getUrl());
                            removeOldEntry = true;
                        } else {
                            boolean failedLastTime;
                            if (new Date().getTime() - indexLastModified.getTime() < 86400000L) {
                                mLog.info("Index entry is from the same day. Therefore we have to recrawl but do not index the document.Creating a new one (source=" + docLastModified + "), (index=" + indexLastModified + "): " + rawDocument.getUrl());
                                this.parseDocument(rawDocument, errorLogger);
                                return;
                            }
                            boolean bl = failedLastTime = doc.get("preparation-error") != null;
                            if (!failedLastTime) {
                                mLog.info("Index entry is already up to date (index=" + indexLastModified + "), " + "(source=" + docLastModified + "): " + rawDocument.getUrl());
                                return;
                            }
                            if (!this.mRetryFailedDocs) {
                                mLog.info("Ignoring " + rawDocument.getUrl() + ", because " + "preparation already failed the last time and no retry is wanted.");
                                return;
                            }
                            mLog.info("Retrying preparation of: " + rawDocument.getUrl());
                            removeOldEntry = true;
                        }
                    } else {
                        mLog.info("Index entry has no last-modified field. Creating a new one: " + rawDocument.getUrl());
                        removeOldEntry = true;
                    }
                }
            }
            if (removeOldEntry) {
                this.markForDeletion(doc);
            }
        }
        this.createNewIndexEntry(rawDocument, errorLogger);
    }

    public void createNewIndexEntry(RawDocument rawDocument, ErrorLogger errorLogger) throws RegainException {
        Document doc;
        if (mLog.isDebugEnabled()) {
            mLog.debug("Creating document: " + rawDocument.getUrl());
        }
        if ((doc = this.mDocumentFactory.createDocument(rawDocument, errorLogger)) != null) {
            this.mAddToIndexProfiler.startMeasuring();
            try {
                this.setIndexMode(1);
                this.pluginManager.eventCreateIndexEntry(doc, this.mIndexWriter);
                this.mIndexWriter.addDocument(doc);
                this.mAddToIndexProfiler.stopMeasuring(rawDocument.getLength());
            }
            catch (IOException exc) {
                this.mAddToIndexProfiler.abortMeasuring();
                throw new RegainException("Adding document to index failed", exc);
            }
        }
    }

    public void parseDocument(RawDocument rawDocument, ErrorLogger errorLogger) throws RegainException {
        if (mLog.isDebugEnabled()) {
            mLog.debug("Creating document: " + rawDocument.getUrl() + " only for parsing.");
        }
        this.mDocumentFactory.createDocument(rawDocument, errorLogger);
    }

    public DocumentFactory getDocumentFactory() {
        return this.mDocumentFactory;
    }

    public void removeObsoleteEntries(UrlChecker urlChecker) throws RegainException {
        if (!this.mUpdateIndex) {
            return;
        }
        if (this.mUrlsToDeleteHash == null && urlChecker == null) {
            return;
        }
        UrlMatcher[] preserveUrlMatcherArr = null;
        if (urlChecker != null) {
            preserveUrlMatcherArr = urlChecker.createPreserveUrlMatcherArr();
        }
        this.setIndexMode(2);
        int docCount = this.mIndexReader.numDocs();
        for (int docIdx = 0; docIdx < docCount; ++docIdx) {
            boolean shouldBeDeleted;
            Document doc;
            if (this.mIndexReader.isDeleted(docIdx)) continue;
            try {
                doc = this.mIndexReader.document(docIdx);
            }
            catch (Throwable thr) {
                throw new RegainException("Getting document #" + docIdx + " from index failed.", thr);
            }
            String url = doc.get("url");
            String lastModified = doc.get("last-modified");
            if (url == null) continue;
            if (this.isMarkedForDeletion(doc)) {
                shouldBeDeleted = true;
            } else if (urlChecker == null) {
                shouldBeDeleted = false;
            } else if (urlChecker.shouldBeKeptInIndex(url)) {
                shouldBeDeleted = false;
            } else {
                shouldBeDeleted = true;
                for (int i = 0; i < preserveUrlMatcherArr.length; ++i) {
                    if (!preserveUrlMatcherArr[i].matches(url)) continue;
                    shouldBeDeleted = false;
                    break;
                }
            }
            if (!shouldBeDeleted) continue;
            this.pluginManager.eventDeleteIndexEntry(doc, this.mIndexReader);
            try {
                mLog.info("Deleting from index: " + url + " from " + lastModified);
                this.mIndexReader.deleteDocument(docIdx);
                continue;
            }
            catch (IOException exc) {
                throw new RegainException("Deleting document #" + docIdx + " from index failed: " + url + " from " + lastModified, exc);
            }
        }
        this.mUrlsToDeleteHash = null;
    }

    private void removeObsoleteEntries() throws RegainException {
        this.removeObsoleteEntries(null);
    }

    private void markForDeletion(Document doc) {
        if (this.mUrlsToDeleteHash == null) {
            this.mUrlsToDeleteHash = new HashMap();
        }
        String url = doc.get("url");
        String lastModified = doc.get("last-modified");
        if (url != null || lastModified != null) {
            mLog.info("Marking old entry for a later deletion: " + url + " from " + lastModified);
            this.mUrlsToDeleteHash.put(url, lastModified);
        }
    }

    private boolean isMarkedForDeletion(Document doc) {
        String url = doc.get("url");
        String lastModified = doc.get("last-modified");
        if (url == null || lastModified == null) {
            return true;
        }
        if (this.mUrlsToDeleteHash == null) {
            return false;
        }
        String lastModifiedToDelete = this.mUrlsToDeleteHash.get(url);
        return lastModified.equals(lastModifiedToDelete);
    }

    public int getIndexEntryCount() throws RegainException {
        if (this.mIndexReader != null) {
            return this.mIndexReader.numDocs();
        }
        this.setIndexMode(1);
        return this.mIndexWriter.maxDoc();
    }

    private void prepareBreakpoint() throws RegainException {
        if (this.mUrlsToDeleteHash != null) {
            throw new RegainException("There are still documents marked for deletion. The method removeObsoleteEntires(...) has to be called first.");
        }
        this.setIndexMode(4);
        if (this.mErrorLogStream != null) {
            this.mErrorLogWriter.close();
            try {
                this.mErrorLogStream.close();
            }
            catch (IOException exc) {
                throw new RegainException("Closing error log file failed", exc);
            }
            this.mErrorLogWriter = null;
            this.mErrorLogStream = null;
        }
    }

    public void createBreakpoint() throws RegainException {
        mLog.info("Creating a breakpoint...");
        try {
            this.mBreakpointProfiler.startMeasuring();
            this.removeObsoleteEntries();
            this.prepareBreakpoint();
            File tempDir = new File(this.mBreakpointIndexDir.getAbsolutePath() + "_tmp");
            RegainToolkit.deleteDirectory(tempDir);
            tempDir.mkdir();
            RegainToolkit.copyDirectory(this.mTempIndexDir, tempDir, false);
            this.deleteOldIndex(this.mBreakpointIndexDir);
            if (!tempDir.renameTo(this.mBreakpointIndexDir)) {
                throw new RegainException("Renaming temporary copy directory failed: " + tempDir.getAbsolutePath());
            }
            long breakpointSize = RegainToolkit.getDirectorySize(this.mBreakpointIndexDir);
            this.mBreakpointProfiler.stopMeasuring(breakpointSize);
        }
        catch (RegainException exc) {
            this.mBreakpointProfiler.abortMeasuring();
            throw exc;
        }
    }

    public void close(boolean putIntoQuarantine) throws RegainException {
        try {
            this.setIndexMode(1);
            this.mIndexWriter.optimize();
        }
        catch (IOException exc) {
            throw new RegainException("Finishing IndexWriter failed", exc);
        }
        String[] prefetchFields = this.mConfig.getValuePrefetchFields();
        if (prefetchFields != null && prefetchFields.length != 0) {
            StringBuilder msg = new StringBuilder();
            msg.append("Prefetching destinct field values for: ");
            for (int i = 0; i < prefetchFields.length; ++i) {
                msg.append(i != 0 ? ", " : "");
                msg.append(prefetchFields[i]);
            }
            mLog.info(msg.toString());
            this.setIndexMode(2);
            RegainToolkit.readFieldValues(this.mIndexReader, prefetchFields, this.mTempIndexDir);
        }
        this.prepareBreakpoint();
        this.mDocumentFactory.close();
        if (this.mAnalysisDir != null) {
            File termFile = new File(this.mAnalysisDir.getAbsolutePath() + File.separator + "AllTerms.txt");
            this.writeTermFile(this.mTempIndexDir, termFile);
        }
        File targetDir = putIntoQuarantine ? this.mQuarantineIndexDir : this.mNewIndexDir;
        this.deleteOldIndex(targetDir);
        long deadline = System.currentTimeMillis() + 60000L;
        boolean renameSucceed = false;
        while (!renameSucceed && System.currentTimeMillis() < deadline) {
            renameSucceed = this.mTempIndexDir.renameTo(targetDir);
            try {
                Thread.sleep(100L);
            }
            catch (Exception exc) {}
        }
        if (!renameSucceed) {
            throw new RegainException("Renaming " + this.mTempIndexDir + " to " + targetDir + " failed after " + 60L + " seconds!");
        }
        this.deleteOldIndex(this.mBreakpointIndexDir);
    }

    private void deleteOldIndex(File oldIndexDir) throws RegainException {
        if (oldIndexDir.exists()) {
            File secureDir = new File(oldIndexDir.getAbsolutePath() + "_del");
            if (oldIndexDir.renameTo(secureDir)) {
                RegainToolkit.deleteDirectory(secureDir);
            } else {
                throw new RegainException("Deleting old index failed: " + oldIndexDir.getAbsolutePath());
            }
        }
    }

    private void writeTermFile(File indexDir, File termFile) throws RegainException {
        IndexReader reader = null;
        FileOutputStream stream = null;
        PrintWriter writer = null;
        try {
            reader = IndexReader.open(FSDirectory.open(indexDir));
            stream = new FileOutputStream(termFile);
            writer = new PrintWriter(stream);
            writer.println("This file was generated by the crawler and contains all terms in the index.");
            writer.println("It's no error when endings like 'e', 'en', and so on are missing.");
            writer.println("They have been cuttet by the GermanAnalyzer and will be cut from a search query too.");
            writer.println();
            TermEnum termEnum = reader.terms();
            int termCount = this.writeTermsSorted(termEnum, writer);
            mLog.info("Wrote " + termCount + " terms into " + termFile.getAbsolutePath());
        }
        catch (IOException exc) {
            throw new RegainException("Writing term file failed", exc);
        }
        finally {
            if (reader != null) {
                try {
                    reader.close();
                }
                catch (IOException exc) {}
            }
            if (writer != null) {
                writer.close();
            }
            if (stream != null) {
                try {
                    stream.close();
                }
                catch (IOException exc) {}
            }
        }
    }

    private int writeTermsSimply(TermEnum termEnum, PrintWriter writer) throws IOException {
        int termCount = 0;
        while (termEnum.next()) {
            Term term = termEnum.term();
            writer.println(term.text());
            ++termCount;
        }
        return termCount;
    }

    private int writeTermsSorted(TermEnum termEnum, PrintWriter writer) throws IOException {
        ArrayList<String> list = new ArrayList<String>();
        while (termEnum.next()) {
            Term term = termEnum.term();
            list.add(term.text());
        }
        Object[] asArr = new String[list.size()];
        list.toArray(asArr);
        Arrays.sort(asArr);
        for (int i = 0; i < asArr.length; ++i) {
            writer.println((String)asArr[i]);
        }
        return asArr.length;
    }
}

